#!/usr/bin/perl -- # -*- mode:cperl; cperl-indent-level:4; cperl-continued-statement-offset:4; indent-tabs-mode:nil; fill-column:80 -*-
# vim: set ts=4 sts=4 sw=4 et:

# Copyright (C) 2003, 2004 Verotel, Inc
# All Rights Reserved
#
# This script uses Verotel RUM Protocol version 1.1
#
# $Rev: 11400 $
# $Date: 2013-10-02 $

use File::Copy;

use strict;
use warnings FATAL => 'all';
use diagnostics;

# {{{ Docs & configuration

=head1 NAME

verotel_RUM_ht - Remote User Management (version 1.1) service for use with
C<htpasswd> files

=head1 DESCRIPTION

This script handles access account management (adding, modifying, deleting) for
Apache web server. User accounts are stored in C<htpasswd> file.

New user accounts are created with C<add> call. User's password is encrypted
before it's sent over the Internet. If your server is unreachable at the time of
C<add> call, Verotel will go on trying to add the account for another half an
hour. If new account cannot be created within 30 minutes after customer was
charged, Verotel will refund the customer and notify both web-master and
customer by email.

C<add> call will contain following data: usercode, passcode (encrypted),
transaction amount (in USD) and optionally Verotel unique transaction number.

Web-master can change password of existing users in Verotel Control Center. In
this case C<modify> call is issued. It contains usercode and encrypted password.

User accounts are removed from system by C<delete> call. This call only contains
usercode.

Users can indicate they don't want their recurring subscriptions to be re-billed
again. Once they do so (by visiting http://cancel.verotel.com), Verotel will
issue C<cancel> call containing usercode. No functionality is provided in this
script for C<cancel> by default, it's safe to ignore this call. Should you need
some functionality tied to this event, you're free to fill in your own custom
code.

Every time a recurring subscription is re-billed, Verotel will issue C<rebill>
call.  This call will contain usercode, transaction amount (in USD) and unique
transaction number. Again, the default C<rebill> handler is empty and can be
safely ignored, but you're free to fill in custom code.

=head1 INSTALLATION

1. Copy the file C<script.pl> to C<cgi-bin> directory of your website, give it a
name of your choice and set permissions on the file:

  $ cp script.pl /path/to/cgi-bin/whatever_name.pl
  $ chmod 755 /path/to/cgi-bin/whatever_name.pl

2. The password file (C<.htpasswd>) will be stored in so-called data directory
on your website. Either use already existing directory or create a new one and
set permissions on this directory.

  $ mkdir /full/path/to/your_data_dir
  $ chmod 770 /full/path/to/your_data_dir

3. Copy all files from C<data> directory in the distribution file to your newly
created data directory.

  $ cp data/* data/.* /full/path/to/your_data_dir

4. Edit the script and change value of C<$prefix> variable to the name of your
data directory (including full path).

5. [Optional] You can modify other variables in the configuration area to better
suit your needs.

6. Modify C<.htaccess> file in protected area of your website. This file
specifies location of your password file. It should contain line like

  AuthUserFile /full/path/to/your_data_dir/.htpasswd

7. Visit Verotel Control Center and test your RUM setup by trying to add user
account. Check that the added test user account allows you to access protected
area of your website. Try to delete this user account and check access to the
protected area isn't possible for this user anymore.

8. If your RUM setup works fine, please submit a test request so a Verotel
Engineer can test your installation. If your RUM setup doesn't work, please
submit an installation request, and a Verotel Engineer will perform the
installation for you at no cost.

=head1 CONFIGURATION

Keep the configuration files in directories invisible to users browsing your
website. The actual physical location of the files doesn't matter as long as
ordinary users don't have access to them.

=head2 $prefix

This is the data directory which contains the password file, log file and other
supporting files.

Replace default value in C<$prefix> with full path to data directory on your
website. To prevent ordinary users from accessing the directory and files
inside, you should set proper file permissions on them.

  $ chmod 770 /full/path/to/your_data_dir

=cut

our $prefix            = '/full/path/to/your_data_dir';

=head2 $ip_file

To make sure only Verotel servers can call this script, caller's IP address is
checked against list of trusted IP addresses and only connections from IP
address found on this list are allowed. This list is stored in
C<$ip_file>.

IP addresses are stored one per line, in IP/mask (C<A.B.C.D/dd>) format.  Mask
is a number between 1 and 32 (inclusive) and defines network mask length in
bits. Bit mask length of 32 defines single IP address, shorter masks define IP
address ranges (subnets).

Current Verotel IP range is C<195.20.32.128/25>.

Don't forget to set proper file permissions to prevent ordinary users from
reading or writing the file:

  $ chmod 660 /full/path/to/your_data_dir/.ht_IP_addresses

=cut

our $ip_file           = "$prefix/.ht_IP_addresses";

=head2 @file_config

Specify locations of C<htpasswd> files, along with their respective log files.
C<@file_config> consists of <password file - log file> pairs. Most people
will only need one such pair, but should you need the accounts to be mirrored
across several websites, you can list as many pairs as you want.

Set permissions on your C<.htpasswd> and log files.

  $ chmod 660 /full/path/to/your_data_dir/.htpasswd
  $ chmod 660 /full/path/to/your_data_dir/verotel_RUM.log

=cut

our @file_config = (    # passwd file          log file
                    "$prefix/.htpasswd", "$prefix/verotel_RUM.log",
                    #   "passwd file 2", "log file 2",
                    #   "passwd file 3", "log file 3",
                    # and so on...
                    #
                    # Please list the local password and log file first

                    # Or if you want to chain-call another RUM script
                    # running on another server:
                    #
                    #   "http://your.server/cgi/rum_script.pl?", "$prefix/verotel_RUM.log",
                    #
                    # Note that 'http://' at the beginning and '?' at the
                    # end of the URL are mandatory. The log file name is empty.
                    # The local log file $file_config[1] will be used.
                   );

our $suffix = '.tmp';

=head2 $access_lock_file

This file is used as a lock to prevent concurrent changes to C<$storage_file>.

Don't forget to set the same permissions as above.

  $ chmod 660 /full/path/to/your_data_dir/.htlock

=cut

our $access_lock_file  = "$prefix/.htlock";

#------------------------------------------------------------------------------
# End of configuration
#------------------------------------------------------------------------------

=head1 API

Verotel communicates to RUM script via HTTP POST call. The data sent consists of
key-value pairs.

=head2 Call parameters

This is list of parameter which can appear in the call to RUM script:

=over

=item trn

This parameter is always present and mandatory. Specifies action to be taken by
the script.  Possible values: C<add>, C<modify>, C<rebill>, C<cancel>,
C<delete>, C<list_ip>.

=item usercode

User login name.

=item passcode

Encrypted user password.

=item amount

Transaction amount ($$$.cc).

=item trn_id

Unique transaction ID.

=item custom1, custom2, custom3

Custom fields specified by web-master and passed to Verotel via the sign
up button at web-master's site.

=back

=head2 Transaction calls

Following actions can be initiated from Verotel:

=over

=item add

Request addition of new user account. C<trn> is set to C<add>. Parameters
C<usercode>, C<passcode> and C<amount> are always present. Optionally C<trn_id>
is set, too. Eventually all C<add> calls will include C<trn_id>. Any custom
field (C<custom1-3>) set for the particular transaction will be also reported.

Sample C<add> call:

 trn=add&usercode=test&passcode=34kAv5swL0lo.&amount=14.95&trn_id=123456
 trn=add&usercode=test2&passcode=34kAv5swL0lo.&amount=10&trn_id=23455&custom1=foo&custom2=bar

=item modify

Request change of password for existing user account.  C<trn> is set to
C<modify>. C<usercode> and C<passcode> are set.

Sample C<modify> call:

 trn=modify&usercode=test&passcode=Pl8r5Qsg

=item rebill

Recurred transaction notification.  C<trn> is set to C<rebill>. C<usercode>,
C<amount> and C<trn_id> are defined. Any custom field (C<custom1-3>) set for the
particular transaction will be also reported.

Sample C<rebill> call:

 trn=rebill&usercode=test&amount=11&trn_id=2234567
 trn=rebill&usercode=test2&amount=15&trn_id=24589&custom1=foo&custom2=bar

=item cancel

Notification sent after users cancel their account.  C<trn> is set to C<cancel>.
Parameter C<usercode> is always defined.

Sample C<cancel> call:

 trn=cancel&usercode=test

=item delete

Request removal of existing user account.  C<trn> is set to C<delete>.
Parameter C<usercode> is always defined. Any custom field (C<custom1-3>) set for
the particular transaction will be also reported.

Sample C<delete> call:

 trn=delete&usercode=test2&custom1=foo&custom2=bar

=item list_ip

List contents of the IP address file.  C<list_ip>: C<trn> is set to
C<list_ip>. No other parameters are sent.

=back

=head2 Notes on customization

If you need to customize this script (or to rewrite it completely in another
language), please make sure your version adheres to following rules:

1. ALWAYS send back a reply - allowed values are defined below in variables
C<$APPROVED>, C<$DECLINED> and C<$ERROR>

2. C<add> call: don't add duplicate records to .htpasswd file - return
C<$DECLINED> if duplicate is found. Only return C<$APPROVED> after record was
successfully added. Otherwise return C<$ERROR>.

3. C<modify> call: return C<$DECLINED> if given user account doesn't exist in
your system. Return C<$APPROVED> after successful update of your .htpasswd file.
Otherwise return C<$ERROR>.

4. C<delete> call: return C<$DECLINED> if given user account doesn't exist in
your system. Return C<$APPROVED> after given user account is successfully
removed. Otherwise return C<$ERROR>.

5. C<rebill> and C<cancel> calls: always return C<$APPROVED>. Other than that,
these transactions are not required to do anything, the calls are provided for
your convenience, should you want to perform custom actions.

6. There is another call C<list_ip>, which allows Verotel to remotely check your
IP file setup (see L</$ip_file> section below). Please don't modify this
functionality. If you write your own RUM script from scratch, please recreate
this function in its entirety.

=cut

# }}} ------------------------------------------------------------

our $have_log;
our ($LOCK, $LOG);
our $first_ok = 0;
our $APPROVED = 'APPROVED';
our $DECLINED = 'DECLINED';
our $ERROR    = 'ERROR';

my %mandatory_format = (
                        usercode   => q/^[\w_][\w_ ]{0,11}$/,
                        passcode   => q/^[.\/0-9a-zA-Z]{13}$/,
                        amount     => q/^\d{1,3}(\.\d{0,2})?$/,
                        trn_id     => q/^\d{1,12}$/
                       );

my %mandatory_params = (
                        add        => [qw(usercode passcode amount)],
                        modify     => [qw(usercode passcode)],
                        rebill     => [qw(usercode amount trn_id)],
                        cancel     => [qw(usercode)],
                        delete     => [qw(usercode)],

                        list_ip    => [qw()]
                       );


sub trn_add ($);
sub trn_modify ($);
sub trn_delete ($);
sub trn_rebill ($);
sub trn_cancel ($);
sub trn_list_ip;
sub create_lock;
sub destroy_lock;
sub log_msg ($);
sub with_error ($);
sub croak ($);
sub remote_ip_address_allowed;
sub check_ip_format ($);
sub get_cgi_params;
sub call_rum ($);

#------------------------------------------------------------------------------

my %trn_hdl = (
               list_ip    => 'trn_list_ip',
               add        => 'trn_add',
               modify     => 'trn_modify',
               rebill     => 'trn_rebill',
               cancel     => 'trn_cancel',
               delete     => 'trn_delete'
              );

our %params = get_cgi_params;

print "Content-type: text/plain\n\n";

open ($LOG, ">>$file_config[1]") && do { $have_log = 1 };

my $message = "request from IP $ENV{REMOTE_ADDR}";

foreach my $name (sort grep {$_ ne 'passcode'} keys %params) {
    $message .= ", $name '$params{$name}'";
}

log_msg($message);

croak("connection from $ENV{REMOTE_ADDR} denied")
    if ! remote_ip_address_allowed;

croak('invalid or missing transaction type')
    if ! exists $params{trn} || ! $trn_hdl{$params{trn}};

my $trn_type = $params{trn};

foreach my $param (@{$mandatory_params{$trn_type}}) {
    croak("missing parameter $param")
        if ! exists $params{$param};

    croak("invalid parameter $param '$params{$param}'")
        if $params{$param} !~ /$mandatory_format{$param}/;
}

close $LOG;
$have_log = 0;

no strict 'refs';

my $result;

if ($trn_type eq 'list_ip') {
    $result = &{$trn_hdl{$trn_type}}();

    print $result;

    exit;

} else {
    for (my $i = 0; $i < @file_config; $i += 2) {
        my ($storage_file, $log_file) = @file_config[$i, $i+1];

        if ($storage_file =~ /^http:/) {
            open ($LOG, ">>$file_config[1]") && do { $have_log = 1 };

            $result = call_rum($storage_file);

            close $LOG;
            $have_log = 0;

        } else {
            open ($LOG, ">>$log_file") && do { $have_log = 1 };

            $result = &{$trn_hdl{$trn_type}}($storage_file);

            close $LOG;
            $have_log = 0;
        }

        # if the operation on the first htpasswd file succeeds, success
        # will be reported no matter it successive htpasswd files fail
        $first_ok = 1 if ! $first_ok && $result eq $APPROVED;

        last if ! $first_ok;
    }

    print $first_ok ? $APPROVED : $result;

    exit;
}

#------------------------------------------------------------------------------

sub trn_add ($) {
    my $storage_file = shift;

    create_lock;

    if (-f $storage_file) {
        open (my $USR, $storage_file) ||
            return with_error('cannot read password file');

        while (<$USR>) {
            if (/^$params{usercode}:/) {
                close $USR;

                log_msg("WARN user '$params{usercode}' already exists");

                destroy_lock;
                return $DECLINED;
            }
        }

        close $USR;
    }

    open (my $OUT, ">>$storage_file") ||
        return with_error('cannot open password file');

    return with_error('cannot write password file')
        if ! print $OUT "$params{usercode}:$params{passcode}\n";

    close $OUT || return with_error('cannot close password file');

    log_msg("user '$params{usercode}' added");

    destroy_lock;

    $APPROVED;
}

sub trn_modify ($) {
    my $storage_file = shift;

    if (! -f $storage_file) {
        log_msg("WARN password file doesn't exist");
        return $DECLINED;
    }

    create_lock;
    open (my $USR, $storage_file) || return with_error('cannot open password file');

    my @list;
    my $user_exists = 0;

    while (my $line = <$USR>) {
        chomp $line;
        next if ! $line;

        if ($line =~ /^$params{usercode}:/) {
            $user_exists = 1;
        } else {
            push @list, $line;
        }
    }

    close $USR;

    if (! $user_exists) {
        log_msg("WARN user '$params{usercode}' doesn't exist");
        destroy_lock;
        return $DECLINED;
    }

    my $tmp_storage_file = $storage_file . $suffix;
    copy($storage_file, $tmp_storage_file);
    open (my $OUT, ">$tmp_storage_file") ||
        return with_error('cannot open password file');

    foreach my $record (@list, join ':', $params{usercode}, $params{passcode}) {
        return with_error('cannot write temporary file')
            if ! print $OUT "$record\n";
    }

    close $OUT || return with_error('cannot close temporary file');

    rename($tmp_storage_file, $storage_file) ||
        return with_error('cannot rename temporary file');

    log_msg("user '$params{usercode}' updated");

    destroy_lock;

    $APPROVED;
}

sub trn_delete ($) {
    my $storage_file = shift;

    if (! -f $storage_file) {
        log_msg("WARN password file doesn't exist");
        return $DECLINED;
    }

    create_lock;
    open (my $USR, $storage_file) || return with_error('cannot open password file');

    my @list;
    my $user_exists = 0;

    while (my $line = <$USR>) {
        chomp $line;
        next if ! $line;

        if ($line =~ /^$params{usercode}:/) {
            $user_exists = 1;
        } else {
            push @list, $line;
        }
    }

    close $USR;

    if (! $user_exists) {
        log_msg("WARN user '$params{usercode}' doesn't exist");
        destroy_lock;
        return $DECLINED;
    }

    my $tmp_storage_file = $storage_file . $suffix;
    copy($storage_file, $tmp_storage_file);
    open (my $OUT, ">$tmp_storage_file") ||
        return with_error('cannot open temporary file');

    foreach my $record (@list) {
        return with_error('cannot write temporary file')
            if ! print $OUT "$record\n";
    }

    close $OUT || return with_error('cannot close temporary file');

    rename($tmp_storage_file, $storage_file) ||
        return with_error('cannot rename temporary file');

    log_msg("user '$params{usercode}' deleted");

    destroy_lock;

    $APPROVED;
}

# custom code can be placed here, otherwise just reports success
sub trn_rebill ($) {$APPROVED}
sub trn_cancel ($) {$APPROVED}

sub trn_list_ip {
    open (my $IP, $ip_file) || return with_error('cannot read IP file');

    print while <$IP>;

    close $IP;

    $APPROVED;
}

#------------------------------------------------------------------------------

sub create_lock {
    # add timeout here ?
    open ($LOCK, ">$access_lock_file") or do {
        log_msg('cannot open lock file');
        return 0;
    };

    flock ($LOCK, 2) or do {
        log_msg('cannot lock file');
        return 0;
    };

    1;
}

sub destroy_lock {
    flock($LOCK, 8) if $LOCK;
    close $LOCK     if $LOCK;
}

sub log_msg ($) {
    my $message = shift;

    return if ! $have_log;

    # is POSIX installed everywhere by default? strftime would be nice
    my ($sec, $min, $hr, $mday, $mon, $year) = gmtime(time);

    printf $LOG "%4d-%02d-%02d %02d:%02d:%02d [%d] | %s\n",
        $year+1900, ++$mon, $mday, $hr, $min, $sec, $$, $message;
}

sub with_error ($) {
    my $message = shift;

    if ($have_log) {
        my ($sec, $min, $hr, $mday, $mon, $year) = gmtime(time);

        printf $LOG "%4d-%02d-%02d %02d:%02d:%02d [%d] | ERROR %s\n",
            $year+1900, ++$mon, $mday, $hr, $min, $sec, $$, $message;
    }

    destroy_lock;

    $ERROR;
}

sub croak ($) {
    my $message = shift;

    if ($have_log) {
        my ($sec, $min, $hr, $mday, $mon, $year) = gmtime(time);

        printf $LOG "%4d-%02d-%02d %02d:%02d:%02d [%d] | ERROR %s\n",
            $year+1900, ++$mon, $mday, $hr, $min, $sec, $$, $message;
    }

    print $ERROR;

    destroy_lock;

    exit;
}

sub remote_ip_address_allowed {
    my $test_ip = pack('C4', split /\./, $ENV{REMOTE_ADDR});

    open (my $IN, $ip_file) or do {
        log_msg('cannot open IP file');
        return 0;
    };

    while (my $line = <$IN>) {
        chomp $line;
        next if ! $line;

        if (!check_ip_format($line)) {
            log_msg("invalid ip address range in ip_file: $line");
            next;
        }

        my ($ip, $mask) = split /\//, $line;

        $ip   = pack('C4', split /\./, $ip);
        $mask = ~pack('N', 2**(32-$mask)-1);

        if ($ip eq ($test_ip & $mask)) {
            close $IN;
            return 1;
        }
    }

    close $IN;

    0;
}

sub check_ip_format ($) {
    my @ip_range = split /\//, $_[0];
    return if (scalar @ip_range < 2);

    my ($ip, $mask) = @ip_range;

    my @ip_parts = split /\./, $ip;
    return if scalar @ip_parts != 4;

    foreach (@ip_parts) {
        return if ( $_ =~ /\D/ || $_ !~ /\d/ || $_ < 0 || $_ > 255 );
    }

    return if ( $mask =~ /\D/ || $mask !~ /\d/ || $mask < 1 || $mask > 32 );

    1;
}

sub get_cgi_params {
    my ($request, %out);

    if ($ENV{REQUEST_METHOD} eq 'POST') {
        read(STDIN, $request, $ENV{CONTENT_LENGTH});
        $ENV{QUERY_STRING} = $request;
    } else {
        $request = $ENV{QUERY_STRING};
    }

    foreach my $param (split /[&;]/, $request) {
        $param =~ s/\+/ /g;

        my ($key, $value) = split /=/, $param, 2;

        $key   =~ s/%(..)/pack('c', hex($1))/ge;
        $value =~ s/%(..)/pack('c', hex($1))/ge;

        $out{$key} .= "\0" if exists $out{$key};
        $out{$key} .= $value;
    }

    %out;
}

sub call_rum ($) {
    my $url = shift;
    my $response;

    $SIG{ALRM} = sub { die 'timeout'; };

    my $post_params = "trn=$params{trn}";

    foreach my $param_name (grep { ! /^trn$/ } keys %params) {
        $post_params .= join '', '&', $param_name, '=', $params{$param_name};
    }

    alarm 10;

    eval {
        require LWP::UserAgent;
        require HTTP::Request;

        LWP::UserAgent->import();
        HTTP::Request->import();

        my $agent    = LWP::UserAgent->new;
        my $request  = HTTP::Request->new(POST => $url);
        $request->content_type('application/x-www-form-urlencoded');
        $request->content($post_params);

        $response    = $agent->request($request);
    };

    alarm 0;

    if ($@) {
        log_msg("could not call remote RUM script: $@");
        return $ERROR;

    } elsif ($response->is_success) {
        if ($response->decoded_content eq $APPROVED) {
            #log_msg("RUM call succeeded: $url, " . $response->decoded_content);
        } else {
            log_msg("RUM call failed: $url, " . $response->decoded_content);
        }
        return $response->decoded_content;

    } else {
        log_msg("RUM call failed: $url, " . $response->status_line);
        return $ERROR;
    }
}

=head1 BUGS

There are no known bugs at the moment. If you happen to discover one, please
send a report to <merchantsupport@verotel.com>. Please don't forget to mention
version of the script ($Rev: 11400 $) and any modifications you might have
applied.

=head1 VERSION

 Verotel RUM Protocol 1.1

 Revision:    $Rev: 11400 $
 Last change: $Date: 2013-10-02 $

=head1 COPYRIGHT

Copyright (C) 2003, 2004 Verotel, Inc

=cut
